#include <stdio.h>
#include <stdlib.h>
#include <assert.h>
#include <errno.h>
#include <string.h>

#include "map.h"
#include "gmstuff.h"

/*
 * Scan through the map file counting hosts and xbars
 */
void
count_items(struct mapfile *mp)
{
  char buf[80];

  mp->nhost = 0;
  mp->nxbar = 0;

  mp->fp = fopen(mp->name, "r");
  if (mp->fp == NULL) {
    perror(mp->name);
    exit(1);
  }

  while (fgets(buf, sizeof(buf), mp->fp) != NULL) {
    if (strncmp(buf, "h ", 2) == 0) ++mp->nhost;
    else if (strncmp(buf, "s ", 2) == 0) ++mp->nxbar;
  }

  fclose(mp->fp);
  mp->fp = NULL;
}

/*
 * Parse text defining a crossbar
 */
void
parse_xbar(struct mapfile *mp, char *def)
{
  void *rp;
  char buf[80];
  int index;
  int nconn;
  struct xbar *xp;
  int i;
  int xb2;
  int h2;
  int port2;
  int pn;
  char typ;
  char name[64];
  char sxid[16];
  unsigned long xid;
	int rc;

  rc = sscanf(def, "s %s %s\n", sxid, name);
  assert(rc == 2);

  if (sxid[0] == '-' && sxid[1] == '\0') {
    xid = 0;
  } else {
    errno = 0;
    xid = strtoul(sxid, (char **)NULL, 10);
    assert(errno == 0);
  }
  index = symtab_lookup_register(mp->xsyms, name);
  assert(index >= 0 && index < mp->nxbar);

  xp = mp->xbar + index;
  assert(strlen(name) < sizeof(xp->name));
  strcpy(xp->name, name);

  /* next line is number of ports */
  rp = fgets(buf, sizeof(buf), mp->fp);
  assert(rp != NULL);

  rc = sscanf(buf, "%d", &nconn);
  assert(rc == 1);
  assert(nconn >= 0 && nconn <= MAX_XBAR_PORT);

  /* now, nconn lines of connectivity */
  for (i=0; i<nconn; ++i) {
    rp = fgets(buf, sizeof(buf), mp->fp);
    assert(rp != NULL);
    rc = sscanf(buf, "%d %c %s %s %d", &pn, &typ, sxid, name, &port2);
    assert(rc == 5);
    if (sxid[0] == '-' && sxid[1] == '\0') {
      xid = 0;
    } else {
      xid = strtoul(sxid, (char **)NULL, 10);
      assert(xid);
    }
    
    switch (typ) {
      case 's':
        xb2 = symtab_lookup_register(mp->xsyms, name);
	assert(xb2 >= 0 && xb2 <= mp->nxbar);
	assert(port2 >= 0 && port2 < MAX_XBAR_PORT);

	xp->port[pn].conn_type = CONN_XBAR;
	xp->port[pn].conn_port = port2;
	xp->port[pn].ptr.x = mp->xbar + xb2;
	break;

      case 'h':
        h2 = symtab_lookup_register(mp->hsyms, name);

	assert(h2 >= 0 && h2 <= mp->nhost);
	assert(port2 < MAX_HOST_PORT);

	xp->port[pn].conn_type = CONN_HOST;
	xp->port[pn].ptr.hp = &((mp->host + h2)->port[port2]);
	break;

      default:
	assert(0);
	break;
    }
  }
}

/*
 * parse a host entry
 */
void
parse_host(struct mapfile *mp, char *def)
{
  int rc;
  void *rp;
  char buf[80];
  int index;
  struct host *h;
  int port;
  int xbi;
  int xbp;
  int t;
  int i;
  unsigned int mac[6];
  char name[63];
  char sxid[16]; 	/* xbar ID in string format */
  unsigned long int xid;     /* xbar ID */

  rc = sscanf(def, "h - %s\n", name);
  assert(rc == 1);
  index = symtab_lookup_register(mp->hsyms, name);
  assert(index >= 0 && index < mp->nhost);

  h = mp->host + index;
  assert(strlen(name) < sizeof(h->mapname));
  strcpy(h->mapname, name);

  /* number of ports */
  rp = fgets(buf, sizeof(buf), mp->fp);
  assert(rp != NULL);
  h->nport = atoi(buf);

  for (i = 0; i < h->nport; i++) {

    /* a line defining our xbar connection */
    rp = fgets(buf, sizeof(buf), mp->fp);
    assert(rp != NULL);
  
    rc = sscanf(buf, "%d s %s %s %d\n", &port, sxid, name, &xbp);
    assert(rc == 4);
  
    /* For xbar32's, sxid will be the 4-byte integer xbar32 id,
     * in xbar16's it's simply a '-'
     * */
    if (sxid[0] == '-' && sxid[1] == '\0') {
      xid = 0;
    } else {
      errno = 0;
      xid = strtoul(sxid, (char **)NULL, 10);
      assert(errno == 0);
    }
    assert(port < h->nport);
  
    xbi = symtab_lookup_register(mp->xsyms, name);
    assert(xbi >= 0 && xbi < mp->nxbar);
  
    h->port[i].xbar = mp->xbar + xbi;
    h->port[i].xbport = xbp;
    h->port[i].myhost = h;
  }

  /* a line with MAC address */
  rp = fgets(buf, sizeof(buf), mp->fp);
  assert(rp != NULL);

  /* check for old-style map file */
  if (strncmp("number", buf, 6) == 0) {
    rp = fgets(buf, sizeof(buf), mp->fp);
    assert(rp != NULL);
  }

  rc = sscanf(buf, "address %x:%x:%x:%x:%x:%x",
    &mac[0], &mac[1], &mac[2], &mac[3], &mac[4], &mac[5]);

  if (rc != 6) {
    long long bigmac;
    rc = sscanf(buf, "address %llx\n", &bigmac);
    assert(rc == 1);
    mac[0] = (bigmac >> 40) & 0xff;
    mac[1] = (bigmac >> 32) & 0xff;
    mac[2] = (bigmac >> 24) & 0xff;
    mac[3] = (bigmac >> 16) & 0xff;
    mac[4] = (bigmac >> 8) & 0xff;
    mac[5] = bigmac  & 0xff;
  }
  for (i=0; i<6; ++i) {
    h->mac_addr[i] = mac[i];
  }

  /* try to get the hostname from GM, else use MAC address */
  if (get_gm_host_name(h->mac_addr, h->hostname) == -1) {
    sprintf(h->hostname, "%x:%x:%x:%x:%x:%x",
      mac[0], mac[1], mac[2], mac[3], mac[4], mac[5]);
  }

  /* a "hosttype" line */
  rp = fgets(buf, sizeof(buf), mp->fp);
  assert(rp != NULL);

  /* check for old-style map file */
  if (strncmp("gmId", buf, 4) == 0) {
    rp = fgets(buf, sizeof(buf), mp->fp);
    assert(rp != NULL);
  }


  rc = sscanf(buf, "hostType %d\n", &t);
  assert(rc == 1);
  // assert(t == 1);
}


/*
 * Allocate structs for Hosts and Xbars, then parse the file filling them in.
 */
void
parse_items(struct mapfile *mp)
{
  char buf[80];

  mp->fp = fopen(mp->name, "r");
  if (mp->fp == NULL) {
    perror(mp->name);
    exit(1);
  }

  while (fgets(buf, sizeof(buf), mp->fp) != NULL) {
    if (strncmp(buf, "h ", 2) == 0) parse_host(mp, buf);
    else if (strncmp(buf, "s ", 2) == 0) parse_xbar(mp, buf);
  }

  fclose(mp->fp);
  mp->fp = NULL;
}

void
init_structs(struct mapfile *mp)
{
  int i;
  int j;
  struct xbar *xp;

  mp->host = (struct host *) calloc(sizeof(*mp->host), mp->nhost);
  mp->xbar = (struct xbar *) calloc(sizeof(*mp->xbar), mp->nxbar);
  assert(mp->host != NULL && mp->xbar != NULL);

  /* Fill in xbars with indices and disconnected ports */
  for (i=0; i<mp->nxbar; ++i) {
    xp = mp->xbar + i;

    xp->nport = MAX_XBAR_PORT;

    for (j=0; j<xp->nport; ++j) {
      xp->port[j].conn_type = CONN_NULL;
    }
  }
}


/*
 * Scan through all connections, checking for symmetry
 */
void
verify_items(struct mapfile *mp)
{
  struct xbar *xp;
  struct xbar *xp2;
  struct hostport *hp;
  struct host *h;
  int i;
  int p;
  int p2;

  /* check all xbar ports for symmetry */
  for (i=0; i<mp->nxbar; ++i) {
    xp = mp->xbar + i;
    assert(xp->nport != 0);

    for (p=0; p<xp->nport; ++p) {
      switch (xp->port[p].conn_type) {
	case CONN_NULL:
	  break;

	case CONN_HOST:
	  hp = xp->port[p].ptr.hp;
	  if (hp->xbar != xp) {
	    fprintf(stderr, "host pointer asymmetry x=%d, p=%d\n", i, p);
	  }
	  if (hp->xbport != p) {
	    fprintf(stderr, "host port asymmetry x=%d, p=%d\n", i, p);
	  }
	  break;

	case CONN_XBAR:
	  xp2 = xp->port[p].ptr.x;
	  p2 = xp->port[p].conn_port;
	  if (xp2->port[p2].ptr.x != xp) {
	    fprintf(stderr, "port pointer asymmetry x=%d, p=%d\n", i, p);
	    fprintf(stderr, "%s [%d] pts to %s\n",
		xp2->name, p2, xp2->port[p2].ptr.x->name);
	  }
	  if (xp2->port[p2].conn_port != p) {
	    fprintf(stderr, "port index asymmetry x=%d, p=%d\n", i, p);
	  }
	  break;
      }
    }
  }

  /* check all host ports for symmetry */
  for (i=0; i<mp->nhost; ++i) {
    h = mp->host + i;

    for (p = 0; p < h->nport; p++) {
      hp = &(h->port[p]);
      assert(hp->xbar != NULL);
      xp = hp->xbar;

      if (xp->port[hp->xbport].ptr.hp != hp) {
        fprintf(stderr, "host xbar asymmetry x=%d, p=%d\n", i, hp->xbport);
      }
      if (xp->port[hp->xbport].conn_type != CONN_HOST) {
        fprintf(stderr, "host xbar port asymmetry x=%d, p=%d\n", i, hp->xbport);
      }
    }
  }
}

void
load_map_file(struct mapfile *mp)
{
  mp->hsyms = symtab_init();
  mp->xsyms = symtab_init();

  count_items(mp);
  init_structs(mp);
  parse_items(mp);
  verify_items(mp);
  fprintf(stderr, "%d hosts, %d xbars\n", mp->nhost, mp->nxbar);
}
